Option Strict Off
Option Explicit On
Module Oscill
    '                              Oscill-2009
    '
    ' Programma per oscillatore esemplice e spettro di risposta     rev. 05.09
    '
    ' Programma scritto con Visual Basic 2005 Express Edition
    '
    ' Note storiche:
    ' Il programma Oscill.bas era stato creato per motivi didattici e di ricerca
    ' ed era scritto in Quick Basic.
    ' Nel 2003  stato tradotto in Visual Basic 5 ed utilizzato per preparare le figure
    ' inserite nel capitolo 2 del libro di A. Ghersi, P. Lenza, Edifici antisismici
    ' in cemento armato, Flaccovio, Palermo, 2009.
    ' Nel preparare il cd allegato al libro il programma  stato tradotto in Visual Basic
    ' Express 2005, perch questo linguaggio (o la sua versione aggiornata)  scaricabile
    ' gratuitamente dal sito della Microsoft.
    '
    ' Avvertenze:
    ' Il programma consente:
    ' 1.  di ottenere la risposta di un oscillatore semplice elastico od elastoplastico
    '     ad un dato accelerogramma;
    ' 2.  di determinare lo spettro di risposta elastico corrispondente ad un dato
    '     accelerogramma;
    ' 3.  di determinare lo spettro di risposta per duttilit assegnata corrispondente
    '     ad un dato accelerogramma.
    ' Il programma  costituito sostanzialmente dalle sole routine di calcolo, senza
    ' maschere di input o di output. La scelta tra le tre alternative sopra elencate,
    ' nonch (nel caso della prima) tra oscillatore elastico od elastoplastico, e la
    ' definizione di alcuni parametri deve essere effettuata assegnando opportuni valori
    ' a variabili di controllo allinterno del programma principale (Main).
    ' Il programma chiede il nome del file dati che contiene laccelerogramma da utilizzare.
    ' I risultati vengono salvati in un file, che pu essere importato in Excel
    ' per diagrammare i risultati.

    Public Const Elastico As Short = 0
	Public Const ElastoPlastico As Short = 1
    Public Const PI As Double = 3.141592654
    Public Const RispElast As Short = 11
    Public Const RispEP As Short = 12
    Public Const SpettroElast As Short = 2
    Public Const SpettroEP As Short = 3

    Public Sub Main()
        ' ------------------------------- dati da definire -------------------------------
        ' Assegnare alla variapile Oper uno dei valori seguenti:
        '   RispElast       per valutare la risposta di un oscillatore elastico
        '   RispEP          per valutare la risposta di un oscillatore elastoplastico
        '   SpettroElast    per valutare lo spettro di risposta elastica
        '   SpettroEP       per valutare lo spettro di risposta a duttilit assegnata
        Dim Oper As Short = SpettroEP
        ' Assegnare i parametri seguenti:
        Dim T, csi, fy, mu As Single
        T = 1               ' periodo T (per RispElast o RispEP)
        csi = 0.05          ' smorzamento percentuale
        fy = 32264          ' forza limite (solo per RispEP)
        mu = 2              ' duttilit (solo per SpettroEP)
        ' ------------------------------- fine dati da definire --------------------------
        Dim m As Single = 257.49    ' nota: pu essere assegnato un valore qualsiasi alla massa m
        Dim uy As Double
        Dim u_max As Double
        Dim u2y As Double
        Dim ustep As Double
        Dim u2y0 As Double
        Dim PGA_ass As Double
        Dim PGAg As Double
        Dim i As Short
        Dim u2g_m As Double
        Dim j As Short

        Dim nCop As Short
        Dim nAcc As Short
        Dim tg(10000) As Single
        Dim u2g(10000) As Single
        Dim u2gC(20000) As Single
        Dim dt As Single
        Dim u(20000) As Single
        Dim u1(20000) As Single
        Dim u2(20000) As Single
        Dim fs(20000) As Single
        Dim u2_ass(20000) As Single
        Dim nPassi As Single
        Dim Comp As Short
        Dim c, k As Single

        Dim a As New System.Windows.Forms.OpenFileDialog
        Dim F As Short
        Dim Finp As String = ""     ' file che contiene l'accelerogramma
        Dim Fout As String = ""     ' file di output
        a.Filter = "File che contiene l'accelerogramma|*.*"
        a.ShowDialog()
        If a.FileName = "" Then
            MsgBox("Il file che contiene l'accelerogramma non  stato stato indicato", MsgBoxStyle.OkOnly)
            End
        Else
            Finp = a.FileName
            Select Case Oper
                Case 11
                    Fout = Finp & "_1-1.out"
                Case 12
                    Fout = Finp & "_1-2.out"
                Case 2
                    Fout = Finp & "_2.out"
                Case 3
                    Fout = Finp & "_3.out"
            End Select
        End If
        F = FreeFile()
        On Error GoTo NoFileAcceler
        FileOpen(F, Finp, OpenMode.Input)
        On Error GoTo 0
        MsgBox("I risultati vengono messi nel file " & Fout, MsgBoxStyle.OkOnly)
        ' legge l'accelerogramma e lo prepara per l'integrazione al passo
        Call LeggiAccelerogramma(F, 0, 2, 6, 6, 100, nCop, tg, u2g)
        FileClose(F)
        dt = 0.01
        Call PreparaAccelerogramma(nCop, tg, u2g, dt, nAcc, u2gC)
        nPassi = nAcc - 1

        Dim T1000 As Short ' T per 1000
        Dim u_m(120) As Single
        Dim u1_m(120) As Single
        Dim u2_m(120) As Single
        Select Case Oper
            Case 11, 12
                ' --- risposta di un oscillatore
                If Oper = 11 Then Comp = Elastico Else Comp = ElastoPlastico
                c = csi * 4 * PI / T * m
                k = (2 * PI / T) ^ 2 * m
                Call AnalisiAlPasso(nPassi, dt, u2gC, m, c, k, Comp, fy, u, u1, u2, fs)
                Call SalvaRisposta(Fout, nAcc, dt, u2gC, u, u1, u2, fs)
            Case 2
                ' spettro di risposta elastico
                Comp = Elastico
                j = 0
                T1000 = 10 ' 0
                u2g_m = PGA(nAcc, u2gC)
                F = FreeFile()
                FileOpen(F, Fout, OpenMode.Output)
                WriteLine(F, "   T", "spostamento", "velocit", "accelerazione")
                WriteLine(F, 0, 0, "_", u2g_m)
                Do
                    j = j + 1
                    If T1000 < 600 Then
                        T1000 = T1000 + 10 ' 20
                    ElseIf T1000 < 1000 Then
                        T1000 = T1000 + 25 ' 50
                    ElseIf T1000 < 2000 Then
                        T1000 = T1000 + 50 ' 100
                    Else
                        T1000 = T1000 + 100 ' 200
                    End If
                    T = T1000 / 1000
                    c = csi * 4 * PI / T * m
                    k = (2 * PI / T) ^ 2 * m
                    Call AnalisiAlPasso(nPassi, dt, u2gC, m, c, k, Comp, fy, u, u1, u2, fs)
                    For i = 1 To nAcc
                        u2_ass(i) = u2(i) + u2gC(i)
                    Next i
                    u_m(j) = PGA(nAcc, u)
                    u1_m(j) = PGA(nAcc, u1)
                    u2_m(j) = PGA(nAcc, u2_ass)
                    WriteLine(F, T, u_m(j), u1_m(j), u2_m(j))
                Loop Until T1000 = 3000
                FileClose(F)
            Case 3
                ' spettro di risposta a duttilit assegnata
                j = 0
                T1000 = 10 ' 0
                u2g_m = PGA(nAcc, u2gC)
                F = FreeFile()
                FileOpen(F, Fout, OpenMode.Output)
                WriteLine(F, "   T", "spostamento", "velocit", "accelerazione", "u2y")
                WriteLine(F, 0, 0, "_", u2g_m, u2g_m)
                Do
                    j = j + 1
                    If T1000 < 600 Then
                        T1000 = T1000 + 10 ' 20
                    ElseIf T1000 < 1000 Then
                        T1000 = T1000 + 25 ' 50
                    ElseIf T1000 < 2000 Then
                        T1000 = T1000 + 50 ' 100
                    Else
                        T1000 = T1000 + 100 ' 200
                    End If
                    T = T1000 / 1000
                    c = csi * 4 * PI / T * m
                    k = (2 * PI / T) ^ 2 * m
                    PGAg = PGA(nAcc, u2gC)
                    ' analisi preliminare elastica
                    Comp = Elastico
                    Call AnalisiAlPasso(nPassi, dt, u2gC, m, c, k, Comp, fy, u, u1, u2, fs)
                    For i = 1 To nAcc
                        u2_ass(i) = u2(i) + u2gC(i)
                    Next i
                    PGA_ass = Int(CDbl(PGA(nAcc, u2_ass)))
                    ' analisi elastoplastica
                    Comp = ElastoPlastico
                    u2y0 = Int(CDbl(Math.Max(Math.Min(PGA_ass / mu / 2, PGA_ass / mu - 100), 2)))
                    If PGA_ass > 100 Then
                        ustep = 2
                    Else
                        ustep = 1
                    End If
                    For u2y = u2y0 To PGA_ass Step ustep
                        fy = m * u2y
                        Call AnalisiAlPasso(nPassi, dt, u2gC, m, c, k, Comp, fy, u, u1, u2, fs)
                        u_max = PGA(nAcc, u)
                        uy = fy / k
                        If u_max < mu * uy Then Exit For
                    Next u2y
                    '
                    u_m(j) = PGA(nAcc, u)
                    u1_m(j) = PGA(nAcc, u1)
                    u2_m(j) = PGA(nAcc, u2_ass)
                    WriteLine(F, T, u_m(j), u1_m(j), u2_m(j), u2y)
                Loop Until T1000 = 3000
                FileClose(F)
        End Select

        End

NoFileAcceler:
        MsgBox("Il file che contiene l'accelerogramma non  stato trovato", MsgBoxStyle.OkOnly)
        End

    End Sub
	
	' ==================== AnalisiAlPasso =========================
	'
	'  Determina la risposta di un oscillatore semplice ad un assegnato
	'  accelerogramma, col metodo di Newmark
	'
	'  Variabili di ingresso:
	'     nPassi    numero di passi (intervalli temporali)
	'     dt        ampiezza passo
	'     u2g()     accelerazione del terreno (colpasso dt)
	'     m         massa
	'     c         coefficiente di smorzamento
	'     k         rigidezza (elastica)
	'     Comp      comportamento (Elastico o ElastoPlastico)
	'     fy        limite di snervamento (per ElastoPlastico)
	'
	'  Variabili di uscita:
	'     u()       spostamento relativo
	'     u1()      velocit relativa
	'     u2()      accelerazione relativa
	'     fs()      forza di richiamo
	'
	' -------------------------------------------------------------
    Public Sub AnalisiAlPasso(ByRef nPassi As Integer, ByRef dt As Double, ByRef u2g() As Single, ByRef m As Single, ByRef c As Single, ByRef k As Single, ByRef Comp As Object, ByRef fy As Single, ByRef u() As Single, ByRef u1() As Single, ByRef u2() As Single, ByRef fs() As Single)

        Dim u_i As Single ' spostamento relativo u a inizio passo
        Dim u1_i As Single ' velocit relativa u1 a inizio passo
        Dim u2_i As Single ' accelerazione relativa u2 a inizio passo
        Dim fs_i As Single ' forza di richiamo a inizio passo
        Dim u_f As Single ' spostamento relativo u a fine passo
        Dim u1_f As Single ' velocit relativa u1 a fine passo
        Dim u2_f As Single ' accelerazione relativa u2 a fine passo
        Dim fs_f As Single ' forza di richiamo a fine passo
        Dim i As Short
        Const beta As Single = 1 / 6
        Const gamma As Single = 1 / 2 ' accelerazione lineare nel passo
        Const Risoluzione As Short = 1

        Dim a, b As Double
        If Risoluzione = 1 Then
            a = m / dt / beta + c * gamma / beta
            b = m / 2 / beta + c * dt * (gamma / 2 / beta - 1)
        End If

        u_f = 0
        u1_f = 0
        u2_f = -u2g(1)
        fs_f = 0
        For i = 1 To nPassi
            u_i = u_f
            u1_i = u1_f
            u2_i = u2_f
            fs_i = fs_f
            u(i) = u_i
            u1(i) = u1_i
            u2(i) = u2_i
            fs(i) = fs_i
            If Risoluzione = 1 Then
                Call Analisi_Passo_1(m, c, k, Comp, fy, beta, gamma, a, b, dt, u2g(i), u_i, u1_i, u2_i, fs_i, u2g(i + 1), u_f, u1_f, u2_f, fs_f)
            Else
                Call Analisi_Passo_2(m, c, k, Comp, fy, beta, gamma, dt, u2g(i), u_i, u1_i, u2_i, fs_i, u2g(i + 1), u_f, u1_f, u2_f, fs_f)
            End If
        Next i
        i = nPassi + 1
        u(i) = u_f
        u1(i) = u1_f
        u2(i) = u2_f
        fs(i) = fs_f

    End Sub
	
    Private Sub Analisi_Passo_1(ByRef m As Single, ByRef c As Single, ByRef k As Single, ByRef Comp As Short, ByRef fy As Single, ByRef beta As Single, ByRef gamma As Single, ByRef a As Single, ByRef b As Single, ByRef dt As Single, ByRef u2g_i As Single, ByRef u_i As Single, ByRef u1_i As Single, ByRef u2_i As Single, ByRef fs_i As Single, ByRef u2g_f As Single, ByRef u_f As Single, ByRef u1_f As Single, ByRef u2_f As Single, ByRef fs_f As Single)
        ' incognita base: incremento di spostamento nel passo (Chopra)

        Dim du2g As Single ' variaz. acceleraz. terreno nel passo
        Dim du As Single ' variazione di spostamento nel passo
        Dim duEP, fs_fEP As Single

        du2g = u2g_f - u2g_i
        du = (-m * du2g + a * u1_i + b * u2_i) / (k + a / dt)
        fs_f = fs_i + k * du
        If Comp = ElastoPlastico Then
            fs_fEP = Math.Max(Math.Min(fs_f, fy), -fy)
            Do
                fs_f = fs_fEP
                duEP = (-m * du2g + a * u1_i + b * u2_i + fs_i - fs_f) / (a / dt)
                fs_fEP = Math.Max(Math.Min(fs_i + k * du, fy), -fy)
            Loop Until QuasiUguale(fs_f, fs_fEP)
            fs_f = fs_fEP
            du = duEP
        End If
        u_f = u_i + du
        u1_f = du * gamma / dt / beta + u1_i * (1 - gamma / beta) - u2_i * dt * (gamma / 2 / beta - 1)
        u2_f = du / dt ^ 2 / beta - u1_i / dt / beta + u2_i * (1 - 0.5 / beta)

    End Sub
	
	Private Sub Analisi_Passo_2(ByRef m As Object, ByRef c As Object, ByRef k As Object, ByRef Comp As Object, ByRef fy As Object, ByRef beta As Object, ByRef gamma As Object, ByRef dt As Object, ByRef u2g_i As Object, ByRef u_i As Object, ByRef u1_i As Object, ByRef u2_i As Object, ByRef fs_i As Object, ByRef u2g_f As Object, ByRef u_f As Object, ByRef u1_f As Object, ByRef u2_f As Object, ByRef fs_f As Object)
		' incognita base: accelerazione a fine passo (vecchio OSCILL)
		
		Dim du2g As Single ' variaz. acceleraz. terreno nel passo
		Dim du As Single ' variazione di spostamento nel passo
		Dim fs_fEP, u2_fEP, Squil As Single
		Dim iter As Short
		
        u2_f = -m * u2g_f - fs_i - (c + k * dt) * u1_i - (c * dt * (1 - gamma) + k * dt ^ 2 * (0.5 - beta)) * u2_i
        u2_f = u2_f / (m + c * dt * gamma + k * dt ^ 2 * beta)
        u1_f = u1_i + (1 - gamma) * dt * u2_i + gamma * dt * u2_f
        u_f = u_i + dt * u1_i + (0.5 - beta) * dt ^ 2 * u2_i + beta * dt ^ 2 * u2_f
        fs_f = fs_i + k * (u_f - u_i)
        If Comp = ElastoPlastico Then
            fs_fEP = Math.Max(Math.Min(fs_f, fy), -fy)
            iter = 0
            Do
                iter = iter + 1
                fs_f = fs_fEP
                u2_fEP = (-m * u2g_f - fs_f - c * u1_f) / m
                u1_f = u1_i + (1 - gamma) * dt * u2_i + gamma * dt * u2_f
                u_f = u_i + dt * u1_i + (0.5 - beta) * dt ^ 2 * u2_i + beta * dt ^ 2 * u2_f
                fs_fEP = Math.Max(Math.Min(fs_i + k * (u_f - u_i), fy), -fy)
                Squil = m * (u2g_f + u2_fEP) + c * u1_f + fs_fEP
                If iter = 100 Then Stop
            Loop Until System.Math.Abs(Squil) <= 0.01
            fs_f = fs_fEP
            u2_f = u2_fEP
        End If
		
	End Sub
	
	' ==================== LeggiAccelerogramma ====================
	'
	'  Legge le coppie tempo-accelerazione da un file, gi aperto
	'
	'  Variabili di ingresso:
	'     F         buffer del file, gi aperto
	'     dt        incremento di t:
	'                 se  0, t deve essere letto dal file
	'                 se  >0, dal file si leggono solo le accelerazioni
	'     nRigS     numero di righe iniziali da saltare:
	'                 se <0 salta tutte le righe iniziali con *
	'     nCampi    numero di campi (o di coppie, se dt=0) per riga
	'     nCarC     numero di caratteri per campo
	'     u2scale   fattore di scala con cui moltiplicare u2g()
	'
	'  Variabili di uscita:
	'     nCop      numero di coppie tempo-accelerazione
	'     tg()      tempo
	'     u2g()     accelerazione
	'
	' -------------------------------------------------------------
    Public Sub LeggiAccelerogramma(ByRef F As Short, ByRef dt As Single, ByRef nRigS As Short, ByRef nCampi As Short, ByRef nCarC As Short, ByRef u2scale As Single, ByRef nCop As Short, ByRef tg() As Single, ByRef u2g() As Single)

        Dim RigS As Short ' contatore righe da saltare
        Dim L As String ' riga letta
        Dim LenL As Short ' numero di caratteri della riga
        Dim nCopL As Short ' numero di coppie disponibili nella riga
        Dim CopL As Short ' coppia in esame, nella riga
        Dim Cop As Short ' coppia in esame, in totale
        Dim PosIni As Short ' posizione di inizio campo
        Dim tCor As Double ' tempo corrente (per dt>0)

        RigS = nRigS
        Cop = 0
        tCor = 0
        If u2scale = 0 Then u2scale = 1
        Do
            L = LineInput(F)
            If RigS < 0 And Left(L, 1) <> "*" Then RigS = 0
            If RigS > 0 Then
                RigS = RigS - 1
            ElseIf RigS = 0 Then
                LenL = Len(L)
                If dt = 0 Then
                    nCopL = Int(LenL / nCarC / 2)
                Else
                    nCopL = Int(LenL / nCarC)
                End If
                PosIni = 1
                For CopL = 1 To nCopL
                    Cop = Cop + 1
                    If dt = 0 Then
                        tg(Cop) = Val(Mid(L, PosIni, nCarC))
                        PosIni = PosIni + nCarC
                    Else
                        tg(Cop) = tCor
                        tCor = tCor + dt
                    End If
                    u2g(Cop) = Val(Mid(L, PosIni, nCarC)) * u2scale
                    PosIni = PosIni + nCarC
                Next CopL
            End If
        Loop Until EOF(F)
        nCop = Cop

    End Sub
	
	' ==================== PGA ====================================
	'
	'  Determina il valore (assoluto) massimo dell'accelerazione
	'
	'  Variabili di ingresso:
	'     nAcc      numero di valori di accelerazione
	'     u2()      accelerazioni
	'
	' -------------------------------------------------------------
    Public Function PGA(ByRef nAcc As Short, ByRef u2() As Single) As Double

        Dim uMax As Double
        Dim Acc As Short

        uMax = 0
        For Acc = 1 To nAcc
            uMax = Math.Max(uMax, System.Math.Abs(u2(Acc)))
        Next Acc
        PGA = uMax

    End Function
	
	' ==================== PreparaAccelerogramma ==================
	'
	'  Estrae da un accelerogramma i valori dell'accelerazione con un passo assegnato
	'  Nota: il passo dovrebbe essere abbastanza pi piccolo di quello
	'        usato nel campionamento dell'accelerogramma
	'
	'  Variabili di ingresso:
	'     nCop      numero di coppie tempo-accelerazione
	'     tg()      tempo
	'     u2g()     accelerazione
	'     dt        intervallo di t col quale estrarre l'accelerazione
	'
	'  Variabili di uscita:
	'     nAcc      numero di valori di accelerazione ottenuti
	'     u2gC()    accelerazioni, valutate con passo dt costante
	'
	' -------------------------------------------------------------
    Public Sub PreparaAccelerogramma(ByRef nCop As Short, ByRef tg() As Single, ByRef u2g() As Single, ByRef dt As Single, ByRef nAcc As Short, ByRef u2gC() As Single)

        Dim Cop As Short ' coppia di input
        Dim tP, aP As Double ' valori t, acceler di input
        Dim Acc As Short ' coppia predisposta con passo dt costante
        Dim a2, a1, a, T, t1, t2, at As Object

        ' controlli e definizione istante iniziale
        If tg(1) < 0 Then Stop
        If u2g(1) = 0 Then
            t1 = 0
        Else
            t1 = Math.Min(0, tg(1) - 0.02)
        End If
        a1 = 0
        ' lettura e preparazione
        Cop = 1 : tP = tg(Cop) : aP = u2g(Cop)
        t2 = t1 + dt / 2
        Acc = 0
        Do
            T = t1
            a = a1
            at = 0
            Do
                If tP < t2 Then
                    at = at + (a + aP) * (tP - T) / 2
                    T = tP
                    a = aP
                    If Cop = nCop Then Exit Do
                    Cop = Cop + 1 : tP = tg(Cop) : aP = u2g(Cop)
                Else
                    a2 = a + (aP - a) * (t2 - T) / (tP - T)
                    at = at + (a + a2) * (t2 - T) / 2
                    Exit Do
                End If
            Loop
            If Cop = nCop Then Exit Do
            Acc = Acc + 1
            u2gC(Acc) = at / dt
            If Acc = 1 Then u2gC(Acc) = 2 * u2gC(Acc)
            t1 = t2
            a1 = a2
            t2 = t1 + dt
        Loop
        nAcc = Acc

    End Sub
	
	' ==================== SalvaRisposta ==========================
	'
	'  Salva in un file l'accelerogramma con relativa risposta
	'
	'  Variabili di ingresso:
	'     Fout$     nome del file di output
	'     nAcc      numero di valori di accelerazione ottenuti
	'     dt        ampiezza passo
	'     u2gC()    accelerazioni del terreno, con passo dt costante
	'     u()       spostamento relativo
	'     u1()      velocit relativa
	'     u2()      accelerazione relativa
	'     fs()      forza di richiamo
	'
	' -------------------------------------------------------------
    Public Sub SalvaRisposta(ByRef Fout As String, ByRef nAcc As Short, ByRef dt As Single, ByRef u2gC() As Single, ByRef u() As Single, ByRef u1() As Single, ByRef u2() As Single, ByRef fs() As Single)

        Dim i, F As Short

        F = FreeFile()
        FileOpen(F, Fout, OpenMode.Output)
        WriteLine(F, "  t", "acc.g", "spost", "veloc", "acc.ass", "fs")
        For i = 1 To nAcc
            WriteLine(F, (i - 1) * dt, u2gC(i), u(i), u1(i), u2(i) + u2gC(i), fs(i))
        Next i
        FileClose(F)

    End Sub

    Public Function QuasiUguale(ByRef V1 As Double, ByRef V2 As Double) As Boolean
        If V2 = 0 Then
            If System.Math.Abs(V1) < 0.00001 Then
                QuasiUguale = True
            Else
                QuasiUguale = False
            End If
        Else
            If System.Math.Abs((V1 - V2) / V2) < 0.00001 Then
                QuasiUguale = True
            Else
                QuasiUguale = False
            End If
        End If
    End Function

End Module